
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   PROCNMEA.C - NMEA Message Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode and display NMEA-0183 receiver messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\PROCNMEA.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:48:10   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.7   Jul 09 1997 10:15:42   COLEJ
//   Multiple Changes...
//   
//      Rev 1.5   Feb 25 1997 14:38:52   COLEJ
//   changed method of calculating delta positions
//   using a matrix.  
//   deleted single functions for lat and lon to create
//   a combined function.
//   
//      Rev 1.4   Feb 12 1997 16:05:52   COLEJ
//    
//   
//      Rev 1.3   Aug 15 1996 16:55:02   COLEJ
//   Added option to remove sign on LAT and LON for LLA extraction.
//   
//   
//      Rev 1.2   Aug 14 1996 18:16:10   COLEJ
//   Took out local variable in show_utc_time for utc_time.
//   
//   
//      Rev 1.1   Aug 13 1996 15:07:22   COLEJ
//    
//   
//      Rev 1.0   13 May 1996 14:53:24   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>  
#include <stdlib.h>
#include <string.h> 
#include <math.h>
#include <graph.h> 
#include <ctype.h> 
#include <conio.h> 

#include "GENCONS.H"
#include "GENTYPES.H"
#include "ZTYPES.H"
#include "PROCNMEA.H"
#include "LABMON.H" 
#include "DISPLAY.H"
#include "MONSHOW.H"
#include "MONMISC.H" 
#include "SENDNMEA.H"
#include "MENUS.H"
#include "UTIL.H"

#define  MAXCHN 12  

float                hdop; 
static double        utc_time, old_utc_time;  // old utc time is to check for new data on xtract
static short         utc_valid;
extern double        lat;
extern double        lon;
extern double        alt;
static char          latdir;
static char          londir;     
static short         GGA = 0;                 // flag for current msg is an gga

short                nsats_used;
short                gps_qual = 0;
static short         nprns_used; 
static short         sats_used[MAXCHN];                      
static short         num_vis;
static short         sat_vis_id[MAXCHN];
static float         sat_vis_el[MAXCHN];
static float         sat_vis_az[MAXCHN]; 
static short         sat_vis_cn[MAXCHN];       
extern short         delta; 
extern short         filter_on;      
extern short         cmdline;
static short         prwizch_found = 0; 

extern double        ref_lat, ref_lon, ref_alt;
            
extern char          buff[80]; 
extern char          nmea_msg[80];                       
extern double        delta_lat, delta_lon, delta_alt; 
extern unsigned long cntncs, cntovf,  cntalm, cntgga, cntgll, cntgsa ; 
extern unsigned long cntgsv, cntrmc,  cntvtg, cntzda, cntalt, cntcom ;
extern unsigned long cntdgp, cntinit, cntibit, cntzch, cnterr, cntrid, cntbit ;
extern unsigned long CntNav, CntPnt;

extern short         MainBackColor;
extern short         MenuBackColor;
extern short         MenuDescColor;
extern short         MainDataColor;  

extern tDATATYPE     DataType;

extern FILE          *stream3; 

tGeoPos GeoPos, RefGeoPos; // strutures for lat/lon/alt
double  ENU[3];            // used for delta positions  


void process_nmea(void)
{
   extern unsigned char msgbuff[MAX_MSG_BUFFER];
   extern short         xtract;  
   extern char          xtractid[4];
   extern tBOOL         ABS_LLA;  

   extern float  maxhdop;
   extern short  minsats;
   extern short  minfom;
   extern short  minqual;
 
   char *tmp[30];
   short numfield;  
 
   // extract the fields and process by type
   numfield = extract(msgbuff+1,tmp,30);   
                   
   if(strcmp(tmp[0],"GPALM") == 0){
      proc_alm(tmp);
   }   
   else if(strcmp(tmp[0],"GPGGA") == 0){
      proc_gga(tmp);
   }
   else if(strcmp(tmp[0],"GPGLL") == 0){
      proc_gll(tmp);
   }   
   else if(strcmp(tmp[0],"GPGSA") == 0){ 
      proc_gsa(tmp);
   }
   else if(strcmp(tmp[0],"GPGSV") == 0){
      proc_gsv(tmp);
   }   
   else if(strcmp(tmp[0],"GPRMC") == 0){
      proc_rmc(tmp);
   }
   else if(strcmp(tmp[0],"GPVTG") == 0){
      proc_vtg(tmp);
   }  
   else if(strcmp(tmp[0],"GPZDA") == 0){
      proc_zda(tmp);
   }   
   else if(strcmp(tmp[0],"PRWIALT") == 0){
      proc_alt(tmp);
   }
   else if(strcmp(tmp[0],"PRWIBIT") == 0){
      proc_bit(tmp);
   }
   else if(strcmp(tmp[0],"PRWICOM") == 0){
      proc_com(tmp);
   }
   else if(strcmp(tmp[0],"PRWIDGP") == 0){
      proc_dgp(tmp);
   }
   else if(strcmp(tmp[0],"PRWIZCH") == 0){
      proc_zch(tmp);
   } 
   else if(strcmp(tmp[0],"PRWIERR") == 0){
      proc_err(tmp);
   } 
   else if(strcmp(tmp[0],"PRWIRID") == 0){
      proc_rid(tmp);
   } 

   
   // Extraction code

   // load tGeoPos structs with reference terms in degrees
   RefGeoPos.lat = ref_lat;
   RefGeoPos.lon = ref_lon;
   RefGeoPos.alt = ref_alt;
   
   GeoidToENU( GeoPos, RefGeoPos, ENU );  // call function to calculate delta's
   
   ENU[2] -= ref_alt;  // this converts the change in altitude because ENU[2] is maintained stationary
      
   // filter the data using NMEA mode parameters 
   CntPnt++;
           
   if((gps_qual   >= minqual) &&           
      (hdop       <= maxhdop) &&         
      (nsats_used >= minsats)   ){          
               
      filter_on = 0;   
      CntNav++;

      // write extracted data to a file
      if(xtract && (strcmp(xtractid,"LLA") == 0) && (utc_time != old_utc_time)){            
         old_utc_time = utc_time;  // set old time to current
         if( ABS_LLA )  // wants to remove sign of lat log for plotting
            fprintf(stream3, "%16.9f \t%10.6f \t%11.6f \t%9.2f \t%10.2f \t%11.2f \t%9.2f\n",         
               utc_time, fabs(lon), fabs(lat), alt, ENU[0], ENU[1], ENU[2]);
         else           // wants to leave sign on lat log like display
            fprintf(stream3, "%16.9f \t%10.6f \t%11.6f \t%9.2f \t%10.2f \t%11.2f \t%9.2f\n",         
               utc_time, lon, lat, alt, ENU[0], ENU[1], ENU[2]);
      }     
   }         
   else{         
      filter_on = 1;         
   }
   
   // indicate when filtering is occurring       
   show_filter(filter_on);   
       
   // update the message count   
   if(!menu_on){
      show_count();
   }  
}

void proc_alm(char *msgtmp[20]) 
{
   cntalm++;
   return;
}

void proc_gga(char *msgtmp[20])
{   
   float dgps_age;
   
   GGA = 1;  // set gga flag true
   cntgga++;    
   // utc must be shown as valid or invalid
   utc_valid = show_nmea_qual(msgtmp+6);
   show_nmea_utc(msgtmp+1);    
   
   show_nmea_lla(msgtmp+2,msgtmp+4,msgtmp+9);
   show_nmea_nsat(msgtmp+7);
   show_nmea_hdop(msgtmp+8);
   dgps_age = (float) atof(msgtmp[13]); 
   sprintf(buff,"%3.0f",dgps_age);  
   if (DataType == NC_NMEA) {
      _settextposition(DSTR,DSTC+3); _outtext(buff);
   }
   else {
      _settextposition(DSTR+1,DSTC-5); _outtext(buff);
   }
   show_nmea_stnid(msgtmp+14); 
   return;
}

void proc_gll(char *msgtmp[20])
{
   extern short hours;
   extern short minutes;
   extern short day;
   extern short month;
   extern short year;
   extern short seconds;
   char   *msgalt[4]={"0.0","M","0.0","M"};
   
   cntgll++;
   
   show_nmea_lla(msgtmp+1,msgtmp+3,msgalt);

   // utc must be shown as valid or invalid
   utc_valid = show_nmea_pos_stat(msgtmp+6);     
   show_nmea_utc(msgtmp+5);
   return;
}

void proc_gsa(char *msgtmp[20])
{
   cntgsa++;  
   show_nmea_op_mode(msgtmp+1);
   show_nmea_fix_mode(msgtmp+2);
   show_nmea_sats(msgtmp+3);
   show_nmea_pdop(msgtmp+15);
   show_nmea_hdop(msgtmp+16);
   show_nmea_vdop(msgtmp+17);
   return;
}

void proc_gsv(char *msgtmp[20])
{  
   short max_msg = 0; 
   short num_msg = 0;
   
   cntgsv++;  
   max_msg = atoi(msgtmp[1]);  
   num_msg = atoi(msgtmp[2]);
   num_vis = atoi(msgtmp[3]);

   get_nmea_vis_sats(num_msg, msgtmp+4);  
   //if(num_msg == max_msg){
      show_nmea_vis_sats(num_vis); 
      if(!prwizch_found){
         show_nmea_chn_stat(num_vis); 
      }
   //}
   return;
}
   
void proc_rmc(char *msgtmp[20])
{
   char *msgalt[4] = {"0.0","M","0.0","M"};
   
   cntrmc++;          
   // utc must be shown as valid or invalid
   utc_valid = show_nmea_pos_stat(msgtmp+2);  
   show_nmea_utc(msgtmp+1);       

   show_nmea_lla(msgtmp+3,msgtmp+5,msgalt);
   show_nmea_sog(msgtmp+7);
   show_nmea_cog(msgtmp+8);
   show_nmea_date(msgtmp+9);
   show_nmea_mag(msgtmp+10);
   return;
}

void proc_vtg(char *msgtmp[20])
{
   cntvtg++;
   show_nmea_cog(msgtmp+1);
   show_nmea_sog(msgtmp+5);
   return;
}

void proc_zda(char *msgtmp[20])
{
   cntzda++;   
   show_nmea_utc(msgtmp+1); 
   show_nmea_dmy(msgtmp+2);
   return;
}

void proc_alt(char *msgtmp[20])
{
   cntalt++;  
   show_nmea_alt(msgtmp+1);
   return;
}

void proc_bit(char *msgtmp[20]) 
{
   cntbit++;
   show_nmea_bit(msgtmp+1);
   return;
}

void proc_com(char *msgtmp[20])
{
   cntcom++;  
   show_nmea_com(msgtmp+1);
   return;
}

void proc_dgp(char *msgtmp[20])
{
   cntdgp++;  
   show_nmea_dgp(msgtmp+1); 
   show_nmea_stnid(msgtmp+2);
   return;
} 

void proc_zch(char *msgtmp[30])
{
   cntzch++;  
   prwizch_found = 1;
   show_nmea_zchn_stat(msgtmp+1);
   return;
} 

void proc_err(char *msgtmp[20])
{
   cnterr++;
   show_nmea_err(msgtmp+1);
   return;
}

void proc_rid(char *msgtmp[20])
{
   cntrid++;
   return;
}

void get_nmea_vis_sats(short num_msg, char *msgtmp[20])
{   
   short i = 0;
   short j = 0;
   short k = 0;     
   
   k = (num_msg-1) * 4;
   for(i=0; i<4; i++){
      sat_vis_id[i+k] = atoi(msgtmp[j]);
      sat_vis_el[i+k]  = (float) atof(msgtmp[j+1]);
      sat_vis_az[i+k]  = (float) atof(msgtmp[j+2]); 
      sat_vis_cn[i+k]  = atoi(msgtmp[j+3]); 
      j += 4;
   }
}  

void show_nmea_vis_sats(short num_vis)
{          
   short i;
   short row;
        
   // satellite visibility list
   _settextposition(DOPR-1,DOPC+9);     
   sprintf(buff,"%2d",num_vis);
   _outtext(buff); 
            
   row = VISR;              
   for(i=0; i<num_vis; i++){
      // only show azi and elev when id is available
      if(sat_vis_id[i] != 0){  
         _settextposition(row++,VISC);     
         sprintf(buff,"%2d ",sat_vis_id[i]);
         _outtext(buff);        
         // only show azi and elev when not null which gives zeros
         if(sat_vis_el[i] != 0 && sat_vis_az[i] != 0){
            sprintf(buff," %4.1f %4.0f",sat_vis_el[i],sat_vis_az[i]); 
         }
         else{
            sprintf(buff,"            "); 
         }
         _outtext(buff);
      }     
   }
   // clear out the rest of the list area
   sprintf(buff,"               ");
   while(row<VISR+MAXCHN){
      _settextposition(row++,VISC);
      _outtext(buff);  
   } 
} 

void show_nmea_chn_stat(short num_vis)
{          
   short i;     
   short j;
   short row;
   short track_stat[MAXCHN]; 
   short sat_cn_chn[MAXCHN];
   short sat_id_chn[MAXCHN];
 
   // satellite c/n0               
   for(i=0; i<MAXCHN; i++){
      sat_cn_chn[i] = 0; 
      sat_id_chn[i] = 0;
   }
   for(i=0; i<num_vis; i++){  
      if(sat_vis_cn[i] != 0){    
         track_stat[i] = 1;
         // satellite is being tracked
         if(nprns_used != 0){ 
            // GGA message is on, sort cn0 into channel order
            for(j=0; j<MAXCHN; j++){
               // try to find id in primary list
               if(sat_vis_id[i] == sats_used[j]){
                  track_stat[i] = 2; 
                                
                  // set cn0 for output in channel order   
                  sat_cn_chn[j] = sat_vis_cn[i];
                  sat_id_chn[j] = sat_vis_id[i]; 
                  break;             
               }            
            }
         }
      }
      else{
         track_stat[i] = 0;
      }
   }                 
   
   // list primary channel c/n0, need GGA to determine primary sats   
   row = CHNR;
   for(j=0; j<nprns_used; j++){
      _settextposition(row++,CHNC+3); 
      sprintf(buff," %2d        %2d",sat_id_chn[j],sat_cn_chn[j]);
      _outtext(buff);       
   }  
   // list secondary channel c/n0, all data is secondary without GGA
   for(i=0; i<num_vis; i++){  
      if(track_stat[i] == 1){
         _settextposition(row++,CHNC+3); 
         sprintf(buff,"*%2d        %2d",sat_vis_id[i], sat_vis_cn[i]);
         _outtext(buff);     
      }
   }
   sprintf(buff,      "               ");
   while(row<CHNR+MAXCHN){
      _settextposition(row++,CHNC+3);
      _outtext(buff);  
   } 
   
   // clear number of sats used in case GGA message disabled
   nprns_used = 0;
} 

void show_nmea_zchn_stat(char *msgtmp[24])
{          
   short i;     
   short j;
   short row;
   short sat_cn_chn[MAXCHN];
   short sat_id_chn[MAXCHN];   
   short sat_stat_chn[MAXCHN]; 
     
   unsigned short tmp;
   
   // channel status for all 12
   for(i=0; i<MAXCHN; i++){
      sat_id_chn[i]    = atoi(msgtmp[i*2]);
      sscanf(msgtmp[i*2+1], "%1hx", &sat_stat_chn[i]);   
      
      // try to find id in visibility list  
      sat_cn_chn[i] = 0;
      for(j=0; j<num_vis; j++){ 
         if(sat_id_chn[i] != 0 && sat_id_chn[i] == sat_vis_id[j]){
            // set cn0 for output in channel order
            sat_cn_chn[i] = sat_vis_cn[j]; 
            break;
         }
      }
   }

   // list channel id, status and c/n0
   row=CHNR;
   for(i=0; i<MAXCHN; i++){
      _settextposition(row++,CHNC); 
      sprintf(buff,"%2d  %2d  ",i+1, sat_id_chn[i]);
      _outtext(buff); 
      
      // only show data for channels with satellites assigned     
      if(sat_id_chn[i] != 0){  
         tmp = sat_stat_chn[i];    
         sprintf(buff,"%1d%1d%1d%1d%",   
         (tmp      ) & 0x01,           // used measurement   
         (tmp >>  2) & 0x01,           // valid measurement   
         (tmp >>  1) & 0x01,           // ephemeris available   
         (tmp >>  3) & 0x01 );         // corrections available for DGPS   
         _outtext(buff);            
    
         // only show cn for tracked satellites
         if((sat_stat_chn[i] & 0x0004) != 0){   
            sprintf(buff,"  %2d", sat_cn_chn[i]);
         }
         else{   
            sprintf(buff,"      ");
         }
         _outtext(buff);
      }
   }
}

void show_nmea_utc(char *msgtmp[1])
{    
   short utc_hour;
   short utc_mins;
   float utc_secs;  
//   float utc_time;

   utc_secs = (float) atof(msgtmp[0]+4); msgtmp[0][4] = '\0';
   utc_mins = atoi(msgtmp[0]+2); msgtmp[0][2] = '\0';
   utc_hour = atoi(msgtmp[0]);
   utc_time  = utc_hour * 3600.f;
   utc_time += utc_mins * 60.f;
   utc_time += utc_secs;
         
   _settextposition(TIMR+1,TIMC+4); 
   if(utc_time == 0) return;       // don't output non-data       
   show_time(utc_time);
   _settextposition(TIMR+1,TIMC+4);
   if(utc_valid) _outtext("*"); 
   else _outtext(" ");       
   
   sprintf(buff,"%3d:%02d:%05.2f",  utc_hour, utc_mins, utc_secs);
   _outtext( buff );   
} 

void show_nmea_lla(char *msglat[2],char *msglon[2],char *msgalt[4])
{
   short   i, j, k;
   double  x;
   double  alt_msl;
   double  geoid_sep;
 
   // read latitude, adjust datum and convert to deg/min
   lat = (atof(msglat[0]+2));
   msglat[0][2] = '\0';
   lat /= 60.;                   // convert minutes to degrees
   lat += atof(msglat[0]);       // read degrees and add in
   lat -= (delta_lat * R2D);     // add datum corrections
   //if(lat == 0) return;        // don't output non-data 
   
   latdir = msglat[1][0];        // read direction   
   x = lat;                      // lat in degrees
   if(latdir == 'S'){            // load tGeoPos struct (lat) for delta
     GeoPos.lat = -x;
   }
   else{
     GeoPos.lat = x;
   }                 
   i = (short) x;                // lat whole degrees
   x = (x - i) * 60.0;           // lat in minutes 
   j = (short) x;                // lat in whole minutes
   x = (x - j) * 10000.0;        // lat in ten-thousandths of minutes
   k = (short) x;                // lat in whole ten-thousandths of minutes 

   if(!delta){
      sprintf(buff," %c%2d%02d.%04d'",latdir, i, j, k); // LAT
      _settextposition(LATR,LATC+4); _outtext(buff);     
   }
   
   // read longitude and convert to deg/min   
   lon = atof(msglon[0]+3); msglon[0][3] = '\0';
   lon /= 60.;                   // convert minutes to degrees
   lon += atof(msglon[0]);       // read degrees and add in
   lon -= (delta_lon * R2D);     // add datum corrections
   //if(lon == 0) return;        // don't output non-data  
   
   londir = msglon[1][0];        // read direction     
   x = lon;                      // lon in degrees
   if(londir == 'W'){            // load tGeoPos struct (lon) for delta
     GeoPos.lon = -x;
   }
   else{
     GeoPos.lon = x;
   }                 
   i = (short) x;                // lon whole degrees
   x = (x - i) * 60.0;           // lon in minutes 
   j = (short) x;                // lon in whole minutes
   x = (x - j) * 10000.0;        // lon in ten-thousandths of minutes
   k = (short) x;                // lon in whole ten-thousandths of minutes

   if(!delta){
      sprintf(buff,"%c%3d%02d.%04d'",londir, i, j, k); // LON
      _settextposition(LATR+1,LATC+4); _outtext(buff);      
   }

   // read altitude MSL and convert to altitude in WGS-84  
   alt_msl   = atof(msgalt[0]);
   //if(alt_msl == 0) return;               // don't output non-data
   geoid_sep = atof(msgalt[2]);
   alt       = alt_msl + geoid_sep;  
   GeoPos.alt = alt;             // load tGeoPos struct (alt) for delta

   if(!delta && GGA){  // only disply for gga messages
      sprintf(buff,"    %8.1lf", alt); // ALT 
      _settextposition(LATR+2,LATC+4); _outtext(buff);      
   }

   // output delta lat from reference position 
   if(delta){
     // load tGeoPos structs with reference terms in degrees
     RefGeoPos.lat = ref_lat;
     RefGeoPos.lon = ref_lon;
     RefGeoPos.alt = ref_alt;
   
     GeoidToENU( GeoPos, RefGeoPos, ENU );  // call function to calculate delta's
   
     ENU[2] -= ref_alt;  // this converts the change in altitude because ENU[2] is maintained stationary

     sprintf(buff,"%12.2f ",ENU[1]);
     _settextposition(LATR,LATC+4); _outtext(buff);      

     sprintf(buff,"%12.2f ",ENU[0]);
     _settextposition(LATR+1,LATC+4); _outtext(buff);      

     if(GGA){  // only disply for gga messages
       sprintf(buff,"%12.2f",ENU[2]);  
       _settextposition(LATR+2,LATC+4);_outtext(buff);
     }
   }
   GGA = 0;  // always clear gga flag
}
 
    
short show_nmea_qual(char *msgtmp[1])
{
   // read gps fix quality  
   gps_qual = atoi(msgtmp[0]); 
     
   // only output if more detailed fix info unavailable
   _settextposition(MODR+2,MODC+5);
   if(gps_qual == 1){    
      _outtext("  SPS");      // SPS
   }
   else if(gps_qual == 2){
      _outtext(" DGPS");      
   }
   else if(gps_qual == 3){
      _outtext("  PPS");     
   }
   else{ 
      //_setbkcolor(MenuBackColor);  
      _settextcolor(MenuDescColor);
      _outtext(" NONE");    
      //_setbkcolor(MainBackColor);  
      _settextcolor(MainDataColor);
   }
   return gps_qual;
}

short show_nmea_pos_stat(char *msgtmp[1])
{          
   char  pos_stat;  
   short gps_qual = 0;
   
   // read gps postion status 
   _settextposition(MODR,MODC+5);
   pos_stat = msgtmp[0][0];
   if(pos_stat == ' '){
      gps_qual = 0;
      return gps_qual;       // don't output non-data
   }
   else if(pos_stat == 'V'){   
      gps_qual = 0;
      //_setbkcolor(MenuBackColor);  
      _settextcolor(MenuDescColor);
      _outtext(" NONE"); 
      //_setbkcolor(MainBackColor);  
      _settextcolor(MainDataColor);
   }
   else{    
      gps_qual = 1;
      _outtext("  NAV");
   }
   return gps_qual;
}
  
void show_nmea_op_mode(char *msgtmp[1])
{    
   char op_mode;
   
   // read gps op mode 
   op_mode = msgtmp[0][0];  
   _settextposition(SETR+1,SETC+7);
   if(op_mode == ' ') return;   // don't output non-data 
   if(op_mode == 'A'){
       //_outtext("2");
   }
   else{             
      //_setbkcolor(MenuBackColor);  
      _settextcolor(MenuDescColor);
      //_outtext(" ");   
      //_setbkcolor(MainBackColor);  
      _settextcolor(MainDataColor);
   }
} 

void show_nmea_fix_mode(char *msgtmp[1])
{  
   short        fix_mode; 
   static short fix_mode_prev;
 
   // read gps fix mode    
   fix_mode = atoi(msgtmp[0]);
   
   if(fix_mode == 0) return;       // don't output non-data 
   _settextposition(MODR+1,MODC+5);
   if(fix_mode == 1){
      _outtext(" NONE");
   }
   else if(fix_mode == 2){  
      _outtext("   2D");
   }
   else if(fix_mode == 3){  
      _outtext("   3D");
   }
   // beep for changes in fix mode  
   if(fix_mode != fix_mode_prev){
      cprintf("\a");  
   }
   fix_mode_prev = fix_mode;
} 
       
void show_nmea_nsat(char *msgtmp[1])
{ 
   // read number of sats used  
   nsats_used = atoi(msgtmp[0]);
   if(nsats_used == 0) return;       // don't output non-data  
   sprintf(buff,"  %2d",nsats_used); 
   _settextposition(DOPR-2,DOPC+7);
   _outtext(buff);
} 

void show_nmea_sats(char *msgtmp[MAXCHN])
{
   short i;
   
   // read id of sats used  
   nprns_used = 0;
   for(i=0; i<MAXCHN; i++){
      sats_used[i] = atoi(msgtmp[0+i]);
      
      if(sats_used[i] == 0){
         return;
      } 
      else{  
         if(!prwizch_found){
            sprintf(buff,"%2d", sats_used[i]); 
            _settextposition(CHNR+i,CHNC+4);
            _outtext(buff);  
         }
         nprns_used++;
      }  
      sprintf(buff,"  %2d", nprns_used); 
      _settextposition(DOPR-2,DOPC+7);
      _outtext(buff); 
   }
} 
   
void show_nmea_pdop(char *msgtmp[1])
{
   float pdop;
   
   // read pdop
   pdop = (float) atof(msgtmp[0]);
   if(pdop == 0) return;               // don't output non-data
   sprintf(buff,"%6.2f", pdop);                    
   _settextposition(DOPR+2,DOPC+5);
   _outtext(buff);
} 
void show_nmea_hdop(char *msgtmp[1])
{  
   // read hdop
   hdop = (float) atof(msgtmp[0]);
   if(hdop == 0) return;               // don't output non-data
   sprintf(buff,"%6.2f", hdop);                    
   _settextposition(DOPR+3,DOPC+5);
   _outtext(buff);
} 

void show_nmea_vdop(char *msgtmp[1])
{
   float vdop;
   
   // read vdop
   vdop = (float) atof(msgtmp[0]);
   if(vdop == 0) return;               // don't output non-data
   sprintf(buff,"%6.2f", vdop);                    
   _settextposition(DOPR+4,DOPC+5);
   _outtext(buff);
} 

void show_nmea_alt(char *msgtmp[2])
{          
   double alt_msl;
   double geoid_sep;
   double del_alt;
   
   // read altitude MSL and convert to altitude in WGS-84  
   alt_msl   = atof(msgtmp[0]);
   //if(alt_msl == 0) return;               // don't output non-data
   geoid_sep = atof(msgtmp[2]);
   alt       = alt_msl + geoid_sep;  
   
   if(!delta){
      sprintf(buff,"    %8.1lf", alt);  
   }
   else{
      // output delta lat/lon/alt from reference position
      del_alt = alt - ref_alt;
      sprintf(buff,"%12.2f",del_alt);  
   }            
   _settextposition(LATR+2,LATC+4);_outtext(buff);
} 

void show_nmea_bit(char *msgtmp[11])
{
   unsigned short ROMFail, RAMFail, EEPFail;
   unsigned short DPRFail, DSPFail, RTCFail;
   unsigned short SP1Err, SP2Err, SP1Rcv, SP2Rcv;
   float          SWVer;
   char           Buf[80] = "";
     
   extern short BITTest; 
   extern short current_menu;  
   extern short MainTextColor;   
   extern short MainDataColor;

   // special processing
   ROMFail = (unsigned short) strtol(msgtmp[0], NULL, 16);
   RAMFail = (unsigned short) strtol(msgtmp[1], NULL, 16);
   EEPFail = (unsigned short) strtol(msgtmp[2], NULL, 16);
   DPRFail = (unsigned short) strtol(msgtmp[3], NULL, 16);
   DSPFail = (unsigned short) strtol(msgtmp[4], NULL, 16);
   RTCFail = (unsigned short) strtol(msgtmp[5], NULL, 16);

   SP1Err  = (unsigned short) atoi(msgtmp[6]);
   SP2Err  = (unsigned short) atoi(msgtmp[7]);
   SP1Rcv  = (unsigned short) atoi(msgtmp[8]);
   SP2Rcv  = (unsigned short) atoi(msgtmp[9]);

   SWVer   = (float) atof(msgtmp[10]);

   // clear screen and output titles
   _clearscreen(_GCLEARSCREEN);  
   _settextcolor(MainTextColor); 
    
   sprintf(Buf,"ZODIAC BUILT-IN TEST RESULTS"  );ShowText(Buf,BITR- 1,BITC+ 8);
   sprintf(Buf,"FUNCTIONAL TEST"               );ShowText(Buf,BITR+ 1,BITC   ); 
   sprintf(Buf,"FAILURES"                      );ShowText(Buf,BITR+ 1,BITC+37); 
                                                                                                   
   sprintf(Buf,"READ-ONLY MEMORY (ROM)"        );ShowText(Buf,BITR+ 3,BITC   );
   sprintf(Buf,"RANDOM ACCESS MEMORY (RAM)"    );ShowText(Buf,BITR+ 4,BITC   );
   sprintf(Buf,"NON-VOLATILE STORAGE (EEPROM)" );ShowText(Buf,BITR+ 5,BITC   );
   sprintf(Buf,"DUAL PORT RAM (DPRAM)"         );ShowText(Buf,BITR+ 6,BITC   );
   sprintf(Buf,"DIGITAL SIGNAL PROCESSOR (DSP)");ShowText(Buf,BITR+ 7,BITC   );
   sprintf(Buf,"REAL-TIME CLOCK (RTC)"         );ShowText(Buf,BITR+ 8,BITC   );
   sprintf(Buf,"SERIAL PORT 1"                 );ShowText(Buf,BITR+ 9,BITC   );
   sprintf(Buf,"SERIAL PORT 2"                 );ShowText(Buf,BITR+10,BITC   );

   sprintf(Buf,"SERIAL PORT 1 RECEIVE COUNT"   );ShowText(Buf,BITR+12,BITC   );
   sprintf(Buf,"SERIAL PORT 2 RECEIVE COUNT"   );ShowText(Buf,BITR+13,BITC   );

   sprintf(Buf,"SOFTWARE VERSION"              );ShowText(Buf,BITR+15,BITC+10);

   
   // output the data items   
   _settextcolor(MainDataColor);
   sprintf(Buf,"%04X",ROMFail                  );ShowText(Buf,BITR+ 3,BITC+39);
   sprintf(Buf,"%04X",RAMFail                  );ShowText(Buf,BITR+ 4,BITC+39);
   sprintf(Buf,"%04X",EEPFail                  );ShowText(Buf,BITR+ 5,BITC+39);
   sprintf(Buf,"%04X",DPRFail                  );ShowText(Buf,BITR+ 6,BITC+39);
   sprintf(Buf,"%04X",DSPFail                  );ShowText(Buf,BITR+ 7,BITC+39);
   sprintf(Buf,"%04X",RTCFail                  );ShowText(Buf,BITR+ 8,BITC+39);
   sprintf(Buf,"%5u" ,SP1Err                   );ShowText(Buf,BITR+ 9,BITC+38);
   sprintf(Buf,"%5u" ,SP2Err                   );ShowText(Buf,BITR+10,BITC+38);

   sprintf(Buf,"%5u" ,SP1Rcv                   );ShowText(Buf,BITR+12,BITC+38);
   sprintf(Buf,"%5u" ,SP2Rcv                   );ShowText(Buf,BITR+13,BITC+38);

   sprintf(Buf,"%.2f",SWVer                    );ShowText(Buf,BITR+15,BITC+27);

   // pause until a key hit and restore screen
   getch();   
   _clearscreen(_GCLEARSCREEN);
   display_menus(&current_menu);   
   show_screen_titles(); 
   BITTest = FALSE;    
} 
    
void show_nmea_com(char *msgtmp[6])
{
   char         auxstat;
   short        auxbaud;
   short        auxdbit;
   short        auxsbit;  
   char         auxprty; 
   short        auxwcnt;  
   extern short cmdline;  
 
   // read receiver auxiliary port communication parameters
   auxstat = msgtmp[0][0]; 
   auxbaud = atoi(msgtmp[1]);
   auxprty = msgtmp[2][0];
   auxprty = toupper(auxprty);
   auxsbit = atoi(msgtmp[3]);
   auxdbit = atoi(msgtmp[4]);
   auxwcnt = atoi(msgtmp[5]);
  
   // change background if parameters are invalid 
   sprintf(buff, "%04i%1c%1i%1i",auxbaud, auxprty, auxdbit, auxsbit);                    
   _settextposition(DSTR+2,DSTC-8);
   if(auxstat != 'A'){ 
      //_setbkcolor(MenuBackColor);  
      _settextcolor(MenuDescColor);
      _outtext(buff);   
      //_setbkcolor(MainBackColor);  
      _settextcolor(MainDataColor);
   }   
   else{
      _outtext(buff); 
   }
}    

void show_nmea_dgp(char *msgtmp[12])
{          
   unsigned short stat_3d;
   unsigned short seq;
   unsigned short z_cnt;
   unsigned short stn_hlth; 
   unsigned short dgps_on = 0;
   float          ehpe;
   float          evpe;
   unsigned short stat_chn[5]; 
   short          i;      
   unsigned short stat_2d;  
   unsigned short tmp;
  
  // read dgps status  
   sscanf(msgtmp[0], "%4hx", &stat_3d);  
   sscanf(msgtmp[2], "%4hx", &tmp);   

   z_cnt = tmp & 0x1FFF; 
   seq   = tmp >> 13;
   stn_hlth = atoi(msgtmp[3]); 
   ehpe    = (float) atof(msgtmp[4])/1000.f;  
   evpe    = (float) atof(msgtmp[5])/1000.f; 
   sscanf(msgtmp[ 6], "%4hx", &stat_chn[0]);     
   sscanf(msgtmp[ 7], "%4hx", &stat_chn[1]); 
   sscanf(msgtmp[ 8], "%4hx", &stat_chn[2]); 
   sscanf(msgtmp[ 9], "%4hx", &stat_chn[3]); 
   sscanf(msgtmp[10], "%4hx", &stat_chn[4]); 
   sscanf(msgtmp[11], "%3hx", &stat_2d);

   sprintf(buff,"%1d", seq);  
   _settextposition(DSTR+1,DSTC-8); _outtext(buff);   
   sprintf(buff,"%4d", z_cnt);         
   _settextposition(DSTR+1,DSTC+2); _outtext(buff);
   sprintf(buff," %1d", stn_hlth);  
   _settextposition(DSTR+2,DSTC+4);  _outtext(buff);
   sprintf(buff,"%04X-%04X", stat_3d, stat_2d);   
   _settextposition(DSTR+3,DSTC-3); _outtext(buff);
   sprintf(buff,"%4.1f", ehpe);  
   _settextposition(HERR  ,HERC);  _outtext(buff);
   sprintf(buff,"%4.1f", evpe);   
   _settextposition(HERR+1,HERC);  _outtext(buff);    

   for(i=0; i<5; i++){
      _settextposition(DGSR+i,DGSC);      
      tmp = stat_chn[i]; 
      sprintf(buff,"%1d%1d%1d%1d%1d%1d%1d%1d%1d",
      (tmp >>  6) & 0x01,
      (tmp >>  7) & 0x01,
      (tmp >>  8) & 0x01,
      (tmp >>  9) & 0x01,
      (tmp >> 10) & 0x01,
      (tmp >> 11) & 0x01,
      (tmp >> 12) & 0x01,
      (tmp >> 13) & 0x01,
      (tmp >> 14) & 0x01);
      _outtext(buff);
   }          
   // read DGPS status
   _settextposition(MODR+3,MODC+5);
   
   dgps_on = !(stat_3d & 0x0001);
   if(dgps_on){      
      _outtext(" DGPS");
   }
   else{
      _outtext("     ");
   }
}

void show_nmea_stnid(char *msgtmp[1])
{  
    signed short stn_id;  
    
    stn_id   = atoi(msgtmp[0]);    
    sprintf(buff,"%4d", stn_id);
    _settextposition(DSTR  ,DSTC-6);
    _outtext(buff);
} 
      
void show_nmea_sog(char *msgtmp[1])
{
   float sog;
   
   // read sog
   sog = (float) atof(msgtmp[0]);
   sprintf(buff,"%6.2f", sog);                    
   _settextposition(SOGR,SOGC+4);
   _outtext(buff);
} 

void show_nmea_cog(char *msgtmp[1])
{
   float cog;
   
   // read cog
   cog = (float) atof(msgtmp[0]);
   sprintf(buff," %5.1f", cog);                    
   _settextposition(SOGR+1,SOGC+4);
   _outtext(buff);
}
 
void show_nmea_mag(char *msgtmp[2])
{
   float   mag;   
   char    dir;
 
   // read magnetic variation   
   mag = (float) atof(msgtmp[0]);
   dir = msgtmp[1][0];      

   sprintf(buff," %c%04.1f",dir, mag); 
   _settextposition(SOGR+2,SOGC+4);_outtext(buff);
} 

void show_nmea_date(char *msgtmp[1])
{ 
   short day;
   short month;
   short year;
   
   // read date  
   year  = atoi(msgtmp[0]+4); msgtmp[0][4] = '\0'; 
   month = atoi(msgtmp[0]+2); msgtmp[0][2] = '\0';  
   if(month == 0) return;         // don't output non-data 
   day   = atoi(msgtmp[0]); 
   sprintf(buff,"%02d/%02d/%02d", month, day, year); 
   _settextposition(TIMR,TIMC+9);
   _outtext(buff);
} 
 
void show_nmea_dmy(char *msgtmp[3])
{   
   short   day;
   short   month;
   short   year; 
   
   // read day, month and year  
   day = atoi(msgtmp[0]);   
   if(day == 0) return;       // don't output non-data 
   month = atoi(msgtmp[1]);
   year  = atoi(msgtmp[2]) % 100;

   sprintf(buff,"%02d/%02d/%02d", month, day, year); 
   _settextposition(TIMR,TIMC+9);
   _outtext(buff);
} 

void show_nmea_err(char *msgtmp[1])
{          
   short   err_class;    
   short   err_number;     
   char    *err_name[6] =
            {"USER MODE EXCEPTION", "EXEC MODE EXCEPTION", "TRAP", 
            "EXECUTIVE ERROR", "ESR ERROR", "USER ERROR",}; 
    
   err_class  = atoi(msgtmp[0]);   
   err_number = atoi(msgtmp[1]);  

   sprintf(buff,"ERR MSG: CLASS: %s NUMBER: %d ADDRESS: %s",
               err_name[err_class], err_number, msgtmp[2]); 
   
   // output the data items   
   clear_message_line();  
   _settextposition(CMDR-1, CMDC);                                                                                              
   _outtext(buff);
}

